<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);
header('Content-Type: application/json; charset=UTF-8');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

include 'conexion.php';

// Determinar el método de la petición (GET, POST, etc.)
$method = $_SERVER['REQUEST_METHOD'];

// Unificar la obtención de datos de entrada
$data = [];
if ($method === 'POST') {
    $data = json_decode(file_get_contents("php://input"), true);
} else {
    $data = $_GET;
}

$action = $data['action'] ?? '';

switch ($action) {
    case 'get_all':
        handleGetAll($conn);
        break;
    case 'add':
        handleAdd($conn, $data);
        break;
    case 'update':
        handleUpdate($conn, $data);
        break;
    case 'delete':
        handleDelete($conn, $data);
        break;
    default:
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Acción no válida.']);
        break;
}

function handleGetAll($conn) {
    try {
        $sql = "SELECT id, nombre, email, tipo FROM usuarios ORDER BY nombre ASC";
        $result = $conn->query($sql);
        $users = [];
        while($row = $result->fetch_assoc()) {
            $users[] = $row;
        }
        echo json_encode(['success' => true, 'users' => $users]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error del servidor: ' . $e->getMessage()]);
    }
}

function handleAdd($conn, $data) {
    $nombre = $data['nombre'] ?? '';
    $email = $data['email'] ?? '';
    $password = $data['password'] ?? '';
    $tipo = $data['tipo'] ?? 'empleado';

    if (empty($nombre) || empty($email) || empty($password)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Todos los campos son obligatorios.']);
        return;
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Formato de email inválido.']);
        return;
    }

    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

    try {
        $sql = "INSERT INTO usuarios (nombre, email, password, tipo) VALUES (?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssss", $nombre, $email, $hashedPassword, $tipo);
        
        if ($stmt->execute()) {
            http_response_code(201);
            echo json_encode(['success' => true, 'message' => 'Usuario creado exitosamente.']);
        } else {
            // Manejar error de email duplicado
            if ($conn->errno == 1062) {
                 http_response_code(409); // Conflict
                 echo json_encode(['success' => false, 'message' => 'El email ya se encuentra registrado.']);
            } else {
                throw new Exception($stmt->error);
            }
        }
        $stmt->close();
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error al crear el usuario: ' . $e->getMessage()]);
    }
}

function handleUpdate($conn, $data) {
    $id = $data['id'] ?? 0;
    $nombre = $data['nombre'] ?? '';
    $email = $data['email'] ?? '';
    $tipo = $data['tipo'] ?? '';
    $password = $data['password'] ?? '';

    if (empty($id) || empty($nombre) || empty($email) || empty($tipo)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Faltan datos para la actualización.']);
        return;
    }

    try {
        if (!empty($password)) {
            // Si se proporciona una nueva contraseña, la actualizamos
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            $sql = "UPDATE usuarios SET nombre = ?, email = ?, tipo = ?, password = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("ssssi", $nombre, $email, $tipo, $hashedPassword, $id);
        } else {
            // Si no, actualizamos todo excepto la contraseña
            $sql = "UPDATE usuarios SET nombre = ?, email = ?, tipo = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("sssi", $nombre, $email, $tipo, $id);
        }

        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Usuario actualizado exitosamente.']);
        } else {
             if ($conn->errno == 1062) {
                 http_response_code(409);
                 echo json_encode(['success' => false, 'message' => 'El email ya pertenece a otro usuario.']);
            } else {
                throw new Exception($stmt->error);
            }
        }
        $stmt->close();
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error al actualizar el usuario: ' . $e->getMessage()]);
    }
}

function handleDelete($conn, $data) {
    $id = $data['id'] ?? 0;

    if (empty($id)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ID de usuario no proporcionado.']);
        return;
    }

    // Opcional: Añadir lógica para no permitir borrar el último 'jefe' o a sí mismo.

    try {
        $sql = "DELETE FROM usuarios WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $id);

        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                echo json_encode(['success' => true, 'message' => 'Usuario eliminado exitosamente.']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Usuario no encontrado.']);
            }
        } else {
            throw new Exception($stmt->error);
        }
        $stmt->close();
    } catch (Exception $e) {
        http_response_code(500);
        // Manejar error de clave foránea si el usuario tiene ventas/gastos
        if (strpos($e->getMessage(), 'foreign key constraint fails') !== false) {
             echo json_encode(['success' => false, 'message' => 'No se puede eliminar: el usuario tiene transacciones asociadas.']);
        } else {
             echo json_encode(['success' => false, 'message' => 'Error al eliminar el usuario: ' . $e->getMessage()]);
        }
    }
}

$conn->close();
?>