<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);
header('Content-Type: application/json; charset=UTF-8');

include 'conexion.php';

// --- Determinar el Rango de Fechas basado en Parámetros ---
$period = isset($_GET['period']) ? $_GET['period'] : 'monthly';
$year = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');
$month = isset($_GET['month']) ? (int)$_GET['month'] : date('m');

switch ($period) {
    case 'daily':
        $startDate = date('Y-m-d 00:00:00');
        $endDate = date('Y-m-d 23:59:59');
        break;
    case 'weekly':
        $startDate = date('Y-m-d 00:00:00', strtotime('-6 days'));
        $endDate = date('Y-m-d 23:59:59');
        break;
    case 'monthly':
    default:
        $startDate = date('Y-m-d H:i:s', mktime(0, 0, 0, $month, 1, $year));
        $endDate = date('Y-m-t H:i:s', mktime(23, 59, 59, $month, 1, $year));
        break;
}

$reportData = [
    'summary' => ['sales' => 0.0, 'sales_bs' => 0.0, 'expenses' => 0.0, 'expenses_bs' => 0.0, 'profit' => 0.0, 'profit_bs' => 0.0, 'transactions' => 0, 'averageTicket' => 0.0, 'profitMargin' => 0.0],
    'salesByMethod' => [],
    'expensesByCategory' => [],
    'dailyTrend' => [],
    'businessInsights' => []
];

try {
    // 1. Rentabilidad y Métricas
    $stmtSales = $conn->prepare("SELECT SUM(monto_total) as total_usd, SUM(monto_total_bs) as total_bs, COUNT(id) as count FROM ventas WHERE fecha_venta BETWEEN ? AND ?");
    $stmtSales->bind_param("ss", $startDate, $endDate);
    $stmtSales->execute();
    $salesResult = $stmtSales->get_result()->fetch_assoc();
    $stmtSales->close();

    $stmtExpenses = $conn->prepare("SELECT SUM(monto) as total_usd, SUM(monto_bs) as total_bs FROM gastos WHERE fecha_gasto BETWEEN ? AND ?");
    $stmtExpenses->bind_param("ss", $startDate, $endDate);
    $stmtExpenses->execute();
    $expensesResult = $stmtExpenses->get_result()->fetch_assoc();
    $stmtExpenses->close();

    $sales_usd = (float)($salesResult['total_usd'] ?? 0);
    $expenses_usd = (float)($expensesResult['total_usd'] ?? 0);
    $sales_count = (int)($salesResult['count'] ?? 0);
    
    $reportData['summary']['sales'] = $sales_usd;
    $reportData['summary']['sales_bs'] = (float)($salesResult['total_bs'] ?? 0);
    $reportData['summary']['expenses'] = $expenses_usd;
    $reportData['summary']['expenses_bs'] = (float)($expensesResult['total_bs'] ?? 0);
    $reportData['summary']['profit'] = $sales_usd - $expenses_usd;
    $reportData['summary']['profit_bs'] = $reportData['summary']['sales_bs'] - $reportData['summary']['expenses_bs'];
    $reportData['summary']['profitMargin'] = $sales_usd > 0 ? ($reportData['summary']['profit'] / $sales_usd) * 100 : 0;
    $reportData['summary']['transactions'] = $sales_count;
    $reportData['summary']['averageTicket'] = $sales_count > 0 ? $sales_usd / $sales_count : 0;

    // 2. Desglose de Ventas por Método de Pago
    $stmtSalesMethod = $conn->prepare("SELECT metodo_pago as method, SUM(monto_total) as amount FROM ventas WHERE fecha_venta BETWEEN ? AND ? GROUP BY metodo_pago ORDER BY amount DESC");
    $stmtSalesMethod->bind_param("ss", $startDate, $endDate);
    $stmtSalesMethod->execute();
    $result = $stmtSalesMethod->get_result();
    $paymentColors = ['PagoMovil' => '#9B59B6', 'Efectivo Bs' => '#00FF87', 'Punto De Venta' => '#FFD93D', 'BioPago' => '#4ECDC4', 'Divisa' => '#FF6B6B'];
    while($row = $result->fetch_assoc()) {
        $amount = (float)$row['amount'];
        $reportData['salesByMethod'][] = ['method' => $row['method'], 'amount' => $amount, 'percentage' => $sales_usd > 0 ? ($amount / $sales_usd) * 100 : 0, 'color' => $paymentColors[$row['method']] ?? '#888'];
    }
    $stmtSalesMethod->close();

    // 3. Desglose de Gastos por Categoría
    $totalExpenses = $reportData['summary']['expenses'];
    $stmtExpensesCat = $conn->prepare("SELECT c.nombre as category, c.color, SUM(g.monto) as amount FROM gastos g JOIN categorias_gastos c ON g.categoria = c.id WHERE g.fecha_gasto BETWEEN ? AND ? GROUP BY g.categoria ORDER BY amount DESC");
    $stmtExpensesCat->bind_param("ss", $startDate, $endDate);
    $stmtExpensesCat->execute();
    $result = $stmtExpensesCat->get_result();
    while($row = $result->fetch_assoc()) {
        $amount = (float)$row['amount'];
        $reportData['expensesByCategory'][] = ['category' => $row['category'], 'amount' => $amount, 'percentage' => $totalExpenses > 0 ? ($amount / $totalExpenses) * 100 : 0, 'color' => $row['color'] ?? '#888'];
    }
    $stmtExpensesCat->close();

    // 4. Tendencia Semanal (Últimos 7 días)
    $trendStartDate = date('Y-m-d', strtotime('-6 days'));
    $daysOfWeek = ['Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb', 'Dom'];
    $salesByDay = array_fill_keys($daysOfWeek, 0);
    $expensesByDay = array_fill_keys($daysOfWeek, 0);
    $dayMap = [1 => 'Dom', 2 => 'Lun', 3 => 'Mar', 4 => 'Mié', 5 => 'Jue', 6 => 'Vie', 7 => 'Sáb'];

    $sqlTrendSales = "SELECT DAYOFWEEK(fecha_venta) as day_num, SUM(monto_total) as total_sales FROM ventas WHERE fecha_venta >= ? GROUP BY day_num";
    $stmt = $conn->prepare($sqlTrendSales);
    $stmt->bind_param("s", $trendStartDate);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) { $salesByDay[$dayMap[$row['day_num']]] = (float)$row['total_sales']; }
    $stmt->close();

    $sqlTrendExpenses = "SELECT DAYOFWEEK(fecha_gasto) as day_num, SUM(monto) as total_expenses FROM gastos WHERE fecha_gasto >= ? GROUP BY day_num";
    $stmt = $conn->prepare($sqlTrendExpenses);
    $stmt->bind_param("s", $trendStartDate);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) { $expensesByDay[$dayMap[$row['day_num']]] = (float)$row['total_expenses']; }
    $stmt->close();

    foreach($daysOfWeek as $day) { $reportData['dailyTrend'][] = ['day' => $day, 'sales' => $salesByDay[$day], 'expenses' => $expensesByDay[$day]]; }
    
    // 5. Análisis del Negocio
    if (!empty($reportData['salesByMethod'])) { $reportData['businessInsights'][] = ['icon' => 'card', 'text' => $reportData['salesByMethod'][0]['method'] . ' es tu método de pago más popular.', 'color' => '#4ECDC4']; }
    if (!empty($reportData['expensesByCategory'])) { $reportData['businessInsights'][] = ['icon' => 'trending-down', 'text' => 'El mayor gasto del periodo fue en ' . $reportData['expensesByCategory'][0]['category'] . '.', 'color' => '#FF6B6B']; }
    if ($reportData['summary']['profitMargin'] > 50) { $reportData['businessInsights'][] = ['icon' => 'trending-up', 'text' => 'Tu margen de ganancia de ' . number_format($reportData['summary']['profitMargin'], 1) . '% es excelente.', 'color' => '#00FF87']; }
    
    $bestDaySales = 0; $bestDayName = '';
    foreach($reportData['dailyTrend'] as $dayData) { if ($dayData['sales'] > $bestDaySales) { $bestDaySales = $dayData['sales']; $bestDayName = $dayData['day']; } }
    if($bestDayName) { $reportData['businessInsights'][] = ['icon' => 'calendar', 'text' => 'Los ' . strtr($bestDayName, ['Lun'=>'Lunes', 'Mar'=>'Martes', 'Mié'=>'Miércoles', 'Jue'=>'Jueves', 'Vie'=>'Viernes', 'Sáb'=>'Sábados', 'Dom'=>'Domingos']) . ' muestran mejores ventas.', 'color' => '#FFD93D']; }

} catch (Exception $e) {
    http_response_code(500);
    $reportData['error'] = 'Error interno del servidor: ' . $e->getMessage();
}

echo json_encode($reportData);
$conn->close();
?>