<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);
header('Content-Type: application/json; charset=UTF-8');

include 'conexion.php';

// --- Parámetros de Paginación y Filtrado ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20; // Número de transacciones por página
$offset = ($page - 1) * $limit;

// Filtros opcionales que la app enviará
$type = isset($_GET['type']) && in_array($_GET['type'], ['sale', 'expense']) ? $_GET['type'] : 'all';
$startDate = isset($_GET['startDate']) ? $_GET['startDate'] : null;
$endDate = isset($_GET['endDate']) ? $_GET['endDate'] : null;

// --- Construcción de la Consulta Dinámica y Segura ---
$baseQuery = "
    FROM (
        (SELECT id, 'sale' as type, monto_total as amount, monto_total_bs as amount_bs, '' as detail, fecha_venta as timestamp, usuario_id FROM ventas)
        UNION ALL
        (SELECT id, 'expense' as type, monto as amount, monto_bs as amount_bs, categoria as detail, fecha_gasto as timestamp, usuario_id FROM gastos)
    ) as t
    JOIN usuarios u ON t.usuario_id = u.id
";

$whereConditions = [];
$params = [];
$types = "";

if ($type !== 'all') {
    $whereConditions[] = "t.type = ?";
    $params[] = $type;
    $types .= "s";
}
if ($startDate) {
    $whereConditions[] = "DATE(t.timestamp) >= ?";
    $params[] = $startDate;
    $types .= "s";
}
if ($endDate) {
    $whereConditions[] = "DATE(t.timestamp) <= ?";
    $params[] = $endDate;
    $types .= "s";
}

if (!empty($whereConditions)) {
    $baseQuery .= " WHERE " . implode(" AND ", $whereConditions);
}

// --- Paginación ---
// Primero, contamos el total de filas que coinciden con los filtros
$countQuery = "SELECT COUNT(*) as total " . $baseQuery;
$stmtCount = $conn->prepare($countQuery);
if ($types) {
    $stmtCount->bind_param($types, ...$params);
}
$stmtCount->execute();
$totalRows = $stmtCount->get_result()->fetch_assoc()['total'];
$stmtCount->close();

// Ahora, obtenemos los datos de la página actual
$dataQuery = "
    SELECT 
        t.id, t.type, t.amount, t.amount_bs, t.timestamp, u.nombre as userName,
        CASE WHEN t.type = 'sale' THEN (SELECT p.nombre FROM detalle_ventas dv JOIN productos p ON dv.producto_id = p.id WHERE dv.venta_id = t.id ORDER BY dv.id ASC LIMIT 1) ELSE t.detail END as first_item_detail,
        CASE WHEN t.type = 'sale' THEN (SELECT COUNT(id) FROM detalle_ventas WHERE venta_id = t.id) ELSE 1 END as item_count
    " . $baseQuery . " ORDER BY t.timestamp DESC LIMIT ? OFFSET ?";

$types .= "ii";
$params[] = $limit;
$params[] = $offset;

$stmt = $conn->prepare($dataQuery);
if ($stmt) {
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $transactions = [];
    while($row = $result->fetch_assoc()) {
        $transactions[] = [
            'id' => $row['id'], 'type' => $row['type'],
            'amount' => (float)$row['amount'], 'amount_bs' => (float)$row['amount_bs'],
            'detail' => $row['first_item_detail'], 'item_count' => (int)$row['item_count'],
            'userName' => $row['userName'], 'time' => date('d/m/Y H:i', strtotime($row['timestamp']))
        ];
    }

    echo json_encode([
        'success' => true,
        'transactions' => $transactions,
        'page' => $page,
        'hasNextPage' => ($offset + count($transactions)) < $totalRows
    ]);

    $stmt->close();
} else {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error en la preparación de la consulta.']);
}

$conn->close();
?>